<?php

/* ###########################################################################
   The KERNAL - 2025

   THESE ARE CUSTOM TOOLS DESIGNED FOR USE WITH THE NASA/JPL HORIZONS API.

   The code can be added to a program internally or externally via the PHP
   'include()' directive.  It can be changed or expanded upon as needed.

   Author   : Jay Tanner - 2025
   Version  : 2460692
   Language : PHP v8.2.12
   License  : Public Domain

   --------------------------------------------------------------------
   KERNAL CUSTOM EPHEMERIS PARSING VARIABLES, FUNCTIONS AND ARGUMENTS.

   UNNEEDED REFERENCES CAN BE REMOVED.    IF ONLY A FEW ARE NEEDED, THEN
   THEY COULD BE COPIED INTO YOUR SCRIPT AND THEN NO INCLUDED FILES FROM
   OUTSIDE WOULD BE NEEDED.

   Get_Horizons_Tech_Data ()
   Special_Horizons_Query (Query)

   Get_Obj_Data           (TargObjID)
   Get_Target_Name        (TargObjID)
   Get_Ephem_Header       (RawEphem)
   Get_CSV_Labels         (RawEphem)
   Get_CSV_Body           (RawEphem)
   Get_Column_Meanings    (RawEphem)
   Split_Days             (RawEphem)
   HMS_to_Hours           (HMSString, Decimals)
   Hours_to_HMS           (Hours, Decimals)
   DMS_to_Deg             (DMSString, Decimals)
   Compass_Symbol         (AzimDeg)
   Lunar_Phase_to_Text    (PhaseAngDeg)
   JD_Num                 (BCADDateStr, JGAMode)
   Inv_JD_Num             (JDNumber, JGAMode)
   Is_Valid_Date_Str      (BCADDateStr)
   ###########################################################################
*/





/*
   ###########################################################################
   This function returns the decimal hours equivalent to the given HMS string.

   Generic. No special error checking is done.

   NO DEPENDENCIES
   ###########################################################################
*/

   function HMS_to_Hours ($HMSString, $Decimals=16)
{
   $HHmmss   = trim($HMSString);
   $decimals = trim($Decimals);

/* ------------------------------------------------
   Account for and preserve any numerical +/- sign.
   Internal work will use absolute values and any
   numerical sign will be reattached the output.
*/
   $NumSign = substr($HHmmss,0,1);
   if ($NumSign == '-')
      {$HHmmss = substr($HHmmss,1,StrLen($HHmmss));}
   else
      {
       if ($NumSign == '+')
          {$HHmmss = substr($HHmmss,1,StrLen($HHmmss));}

       $NumSign = '+';
      }

// ------------------------------------------------------------------
// Replace any colons : with blank spaces and remove any white space.

   $HHmmss = PReg_Replace("/\s+/", " ", Str_Replace(":", " ", $HHmmss));

// ----------------------------------------
// Count the HMS time elements from 1 to 3.

   $n  = 1 + Substr_Count($HHmmss, ' ');

   $hh = $mm = $ss = 0;

/* ----------------------------------------------------------------------
   Collect all given time element values.  They can be integer or decimal
   values. Only counts up to three HMS values and any values beyond those
   are simply ignored.
*/
   for ($i=0;   $i < 1;   $i++)
  {
   if ($n == 1){list($hh)         = PReg_Split("[ ]", $HHmmss);}
   if ($n == 2){list($hh,$mm)     = PReg_Split("[ ]", $HHmmss);}
   if ($n == 3){list($hh,$mm,$ss) = PReg_Split("[ ]", $HHmmss);}
  }

// ------------------------------------------------------------------------
// Compute HMS equivalent in decimal hours to the given number of decimals.

   return $NumSign.(round((3600*$hh + 60*$mm + $ss)/3600,$decimals));

} // End of  HMS_to_Hours(...)





/*
   ###########################################################################
   This function returns an HMS string equivalent to an hours argument rounded
   to the specified number of decimals.

   Generic. No special error checking is done.

   NO DEPENDENCIES
   ###########################################################################
*/

   function Hours_to_HMS ($Hours, $Decimals=0)
{
   $hours = trim($Hours);  $NumSign = ($hours < 0)? '-':'';
   $hours = Str_Replace('+', '', Str_Replace('-', '', $hours));

// ---------------------
// Set working decimals.

   $Q = 32;
   $decimals = floor(abs(trim($Decimals)));
   $decimals = ($decimals > $Q)? $Q : $decimals;
   $decimals = ($decimals <  0)?  0 : $decimals;

// ------------------------------------
// Compute hours,minutes and seconds to
// the specified number of decimals.

   $hh  = bcAdd($hours, '0');
   $min = bcMul('60', bcSub($hours, $hh, $Q),$Q);
   $mm  = bcAdd($min, '0');
   $sec = bcMul('60', bcSub($min, $mm, $Q),$Q);
   $ss  = SPrintF("%1.$decimals"."f", $sec);
          if ($ss < 10){$ss = "0$ss";}

// -------------------------------------------
// Try to account for that blasted 60s glitch.

   if ($ss == 60) {$mm += 1;  $ss = 0;}
   if ($mm == 60) {$hh += 1;  $mm = 0;}

// ------------------------------------------
// Construct and return time elements string.

   $hh = SPrintF("%02d", $hh);
   $mm = SPrintF("%02d", $mm);
   $ss = SPrintf("%1.$decimals"."f", $ss);
         if ($ss < 10){$ss = "0$ss";}

   return "$NumSign$hh:$mm:$ss";

} // End of  Hours_to_HMS (...)





/*
   ###########################################################################
   This function returns decimal degrees equivalent to a DMS string argument
   to the specified number of decimals.

   The Degrees, Minutes and Seconds string values are separated by spaces.

   Generic.  No special error checking is done.

   NO DEPENDENCIES
   ###########################################################################
*/

   function DMS_to_Deg ($DMSString, $Decimals=14)
{
   $DDmmss   = trim($DMSString);
   $decimals = trim($Decimals);

// -----------------------------------
// Account for any numerical +/- sign.

   $NumSign = substr($DDmmss,0,1);
   if ($NumSign == '-')
      {$DDmmss = substr($DDmmss,1,StrLen($DDmmss));}
   else
      {
       if ($NumSign == '+')
          {$DDmmss = substr($DDmmss,1,StrLen($DDmmss));}
           $NumSign = '+';
      }

// -----------------------
// Remove all white space.

   $DDmmss = PReg_Replace("/\s+/", " ", $DDmmss);

// ----------------------------------------
// Count the DMS time elements from 1 to 3.

   $n  = 1 + Substr_Count($DDmmss, ' ');

   $dd = $mm = $ss = 0;

// --------------------------------------
// Collect all given time element values.
// They can be integer or decimal values.

   for ($i=0;   $i < 1;   $i++)
   {
   if ($n == 1){list($dd)         = PReg_Split("[ ]", $DDmmss);}
   if ($n == 2){list($dd,$mm)     = PReg_Split("[ ]", $DDmmss);}
   if ($n == 3){list($dd,$mm,$ss) = PReg_Split("[ ]", $DDmmss);}
   }

// ----------------------------------
// Compute DMS equivalent in degrees.

   return $NumSign.(round((3600*$dd + 60*$mm + $ss)/3600,$decimals));

} // End of  DMS_to_Deg(...)





/*
   ###########################################################################
   This function extracts and returns ONLY target object name text for any
   given solar syetem object ID# or record #, if such data exists.

   Since there are many, many, many thousands of asteroids, there are many of
   them without actual names like Ceres or Vesta, but identified by a variety
   of various catalog symbols that may seem cryptic at first.

   Example: For Target Object ID = 9941;
   Returns: 9941 Iguanodon (1989 CB3)

   NOTE:
   A semicolon (;) at the end of the ID means an asteroid or small body.

   EXAMPLES:

   Given the Target Object ID = 301
   The returned target name string would be:  'Moon (301)'


   Given the Target Object  ID = 301;
   The returned target name string would be:  '301 Bavaria (A890 WA)'

   ----------------------------
   Given the asteroid BodyID #;
   2934;

   The returned target name string would be:
   '2934 Aristophanes (4006 P-L)'

   If no target object name is found, then an error message is returned.

   ###########################################################################
*/

   function Get_Target_Name ($TargObjID)
{
// ===========================================================================
// Read arguments.

   $TargObjID = trim($TargObjID);
   $Command = URLEncode($TargObjID);

// ---------------------------
// Set to current system date.

   $StartDate = date("Y-M-d");

   $From_Horizons_API =
   "https://ssd.jpl.nasa.gov/api/horizons.api?format=text" .
   "&COMMAND='$Command'"    .
   "&OBJ_DATA='NO'"         .
   "&MAKE_EPHEM='YES'"      .
   "&EPHEM_TYPE='OBSERVER'" ;
// ===========================================================================

/* ----------------------------------------------------------
   Send query to Horizons API to obtain the target body name.
*/
   $W = Str_Replace(",\n", " \n", File_Get_Contents($From_Horizons_API));

/* -----------------------------------
   Check if an ephemeris was returned. If
   not, then return an empty string ('').
*/
   if (StrPos($W, '$$SOE') === FALSE) {return trim($W);}

/* ----------------------------------------------------------
   Get the NASA/JPL Horizons target body name string, if any.
   Not every target body may yet have a fixed name or ID, but
   may possibly still have a record # and ephemeris.
*/
   $T = trim($W);
   $i = StrPos($T, 'Target body name:');
   $j = StrPos($T, '}');
   $W = PReg_Replace("/\s+/", " ", trim(substr($T, $i, $j-$i+1)));
   $W = trim(Str_Replace('Target body name:','', $W));
   $i = StrPos($W, '{');

   return trim(substr($W, 0, $i));

} // End of  Get_Target_Name (...)





/*
   ###########################################################################
   This function extracts and returns ONLY the ephemeris parameters text.  If
   the given data does NOT contain an ephemeris, then FALSE is returned.

   The parameters text contains the body ID and ephemeris settings.

   The ephemeris parameters section always begins with the characters:
   'Ephemeris / API_USER'

   Generic.  No special error checking is done.

   NO DEPENDENCIES
   ###########################################################################
*/

   function Get_Ephem_Header($RawEphem)
{
   $T = trim($RawEphem);
   $i = StrPos($T, "\$\$SOE\n");  if ($i === FALSE) {return $i;}
   $i = StrPos($T, 'Ephemeris / API_USER');
   $T = substr($T, $i, StrLen($T));
   $j = StrPos($T, 'Date_');
   $T = substr($T, 0, $j);
   $i = StrPos($T, '(spreadsheet)') + 13;
   $T = substr($T, 0, $i);
   return trim($T) . "\n" . Str_Repeat("=", 80) . "\n";
}






/*
   ###########################################################################
   This function simply returns the basic physical data for any given body ID
   or space mission cataloged by NASA/JPL and also execute special Horizons
   queries.

   The body ID can be a unique name, NASA Body ID # or an SPK ID# DES=xxxxxxx
   or a special info query or directive.

   Generic.  No special error checking is done.

   NO DEPENDENCIES
   ###########################################################################
*/

   function Get_Obj_Data ($TargObjID)
{
// ===========================================================================
// Construct query URL for the NASA/JPL Horizons API.

   $Command = URLEncode(trim($TargObjID));

   $From_Horizons_API =
   "https://ssd.jpl.nasa.gov/api/horizons.api?format=text" .
   "&COMMAND='$Command'" .
   "&MAKE_EPHEM='NO'"    .
   "&OBJ_DATA='YES'"     ;

// ===========================================================================
// Send query to Horizons API to obtain the basic physical data for the given
// body ID as a single long text string with formatting control codes.

   $w = File_Get_Contents($From_Horizons_API);

   return $w;

} // End of  Get_Obj_Data(...)







/*
   ###########################################################################
   This function extracts and returns ONLY the ephemeris CSV labels line.
   If the given data does NOT contain an ephemeris, then FALSE is returned.

   The column header line always begins with the characters: 'Date_'

   Generic.  No special error checking is done.

   NO DEPENDENCIES
   ###########################################################################
*/

   function Get_CSV_Labels ($RawEphemText)
{
   $T = trim($RawEphemText);
   $i = StrPos($T, "\$\$SOE\n");  if ($i === FALSE) {return $i;}
   $T = substr($T,0,$i);
   $T = substr($T, StrPos($T, 'Date_'), StrLen($T));
   $i = StrPos($T, '*');
   $T = RTrim(substr($T, 0, $i));
   return " $T ";
}





/*
   ###########################################################################
   This function can be used to extract any ephemeris table body from all of
   the other extraneous text that surrounds it. An ephemeris table can consist
   of 1 line up to thousands of lines of CSV data.

   The ephemeris CSV data lines are located between two markers:

   $$SOE = Start Of Ephemeris marker
           and
   $$EOE = End Of Ephemeris marker

   Generic.  No special error checking is done.

   NO DEPENDENCIES
   ###########################################################################
*/
   function Get_CSV_Body ($RawEphemText)
{
// ---------------------------------
// Read raw ephemeris text argument.

   $w = trim($RawEphemText);

/* -------------------------------------------------
   Set pointers to start and end of ephemeris table.
   If there is no ephemeris found, then return the
   raw text as-is, nothing is done. It may possibly
   be an error or some other text instead.
*/
   $i = StrPos($w, "\$\$SOE\n"); if ($i === FALSE) {return $w;}
   $j = StrPos($w, "\$\$EOE\n");

/* --------------------------------------------
   Extract ONLY the required ephemeris CSV data
   line(s) from between the Start/End pointers.
*/
   $EphemTableText = trim(substr($w, $i+5, $j-$i-5));

   return (substr($EphemTableText,0,1) == 'b')? $EphemTableText : " $EphemTableText";

} // End of  Get_CSV_Body (...)





/*
   ###########################################################################
   This function extracts and returns ONLY the ephemeris footer text.  If the
   given data does NOT contain an ephemeris, then FALSE is returned.

   The footer text explains the meanings of the ephemeris data columns.

   The ephemeris footer text always begins with the characters:
   'Column meaning:'

   Generic.  No special error checking is done.

   NO DEPENDENCIES
   ###########################################################################
*/

   function Get_Column_Meanings ($RawEphemText)
{
   $T = trim($RawEphemText);
   $i = StrPos($T, '$$SOE');  if ($i === FALSE) {return $i;}
   $i = StrPos($T, 'Column meaning:');
   $T = substr($T, $i, StrLen($T));
   $w = trim($T);
   return "\n" . Str_Repeat('*', 80)."\n".substr($w, 0, StrPos($T, '*******'))
        . Str_Repeat('*', 80);

} // End of  Get_Column_Meanings (...)


/*
  ############################################################################
  This functions splits an ephemeris at at points where the day changes.

  Generic.  No special error checking is done.

  NO DEPENDENCIES
  ############################################################################
*/


   function Split_Days ($MarkedListText)
{
   $T = trim($MarkedListText);

/* ----------------------------------------------------------------------
   If no ephemeris data is found, then return an empty string as an error
   state indicator.
*/
   if (StrPos($T, '$$SOE') === FALSE) {return '';}

// -------------------------------------------------------------------
// Extract ONLY the CSV ephemeris table from a marked raw dated table.

// -------------------------------------------------
// Set pointers to start and end of ephemeris table.

   $i = StrPos($T, '$$SOE');
   $j = StrPos($T, '$$EOE');

/* ------------------------------------------------
   Extract ONLY the required ephemeris data line(s)
   from between the pointers.
*/
   $T = trim(substr($T, $i+5, $j-$i-5));

// -------------------------------
// Store CSV text block in working
// array and count the elements.

   $wArray = PReg_Split("[\n]", $T);
   $wCount = count($wArray);
   $wTable = '';

// --------------------------------------------------------------
// Construct table with different dates separated by blank lines.

   $FirstLine = trim($wArray[0]);

   if (Is_Numeric(substr($FirstLine,0,1))) {$FirstLine = " $FirstLine";}

   for($i=1;   $i < $wCount;   $i++)
  {
   $PrevLine = trim($wArray[$i-1]);
   $CurrLine = trim($wArray[$i-0]);

   if (Is_Numeric(substr($PrevLine,0,1))) {$PrevLine = " $PrevLine";}
   if (Is_Numeric(substr($CurrLine,0,1))) {$CurrLine = " $CurrLine";}

   $PrevDay = substr($PrevLine,10,2);
   $CurrDay = substr($CurrLine,10,2);

/* --------------------------------------------------
   Compare previous and current day numbers to see if
   they are equal.   If not, then insert a blank line
   before printing the subsequent ephemeris line.
*/
   if ($PrevDay == $CurrDay)
      {$wTable .= "$CurrLine\n";}
   else
      {$wTable .= "\n$CurrLine\n";}
  }
   return RTrim("$FirstLine\n$wTable");

} // End of  Split_Days(...)





/*
   ###########################################################################
   This function returns the closest compass symbol corresponding to azimuth
   angle in degrees.  Measured clockwise from North = 0.

   The compass is divided into 16 named
   cardinal zones.

  ---------------------------------------
    AZIMUTH ANGLE CLOCKWISE FROM NORTH

  Symbol     direction        Angle Deg.
  ------  ---------------     ----------
    N     North                  0.0
    NNE   North Northeast       22.5
    NE    Northeast             45.0
    ENE   East Northeast        67.5
    E     East                  90.0
    ESE   East Southeast       112.5
    SE    Southeast            135.0
    SSE   South Southeast      157.5
    S     South                180.0
    SSW   South Southwest      202.5
    SW    Southwest            225.0
    WSW   West Southwest       247.5
    W     West                 270.0
    WNW   West Northwest       292.5
    NW    Northwest            315.0
    NNW   North Northwest      337.5
    N     North                360.0

   Generic.  No special error checking is done.

   NO DEPENDENCIES
   ###########################################################################
*/

   function Compass_Symbol ($AzimDeg)
{
   $a = FloatVal($AzimDeg);

   $a -= 360*floor($a/360);

   $i = ($a < 11.25 or $a > 348.75)? 0 : floor(0.5 + $a/22.5);

   $w = trim(substr('N  NNENE ENEE  ESESE SSES  SSWSW WSWW  WNWNW NNWN', 3*$i, 3));

   $w = substr("$w   ",0,3);
        if (substr($w, -2) == '  ') {$w = ' '.trim($w).' ';}

   return $w;

} // End of  Compass_Symbol (...)




/*
   ###########################################################################
   This function returns a text description of the lunar phase corresponding
   to the given phase angle argument in degrees.

   ------------------------------------------------------
   The simple lunar phase angle = 0 to 360 degrees
   reckoned clockwise where:

     0 = New Moon
    90 = First Quarter Moon
   180 = Full Moon
   270 = Last Quarter Moon
   360 = New Moon

   ERRORS
   Returns FALSE on error if phase angle
   is non-numeric or out of valid range.

   NO DEPENDENCIES
   ###########################################################################
*/

   function Lunar_Phase_to_Text ($PhaseAngDeg)
{
// Round absolute phase angle to nearest integer value.
   $a = trim($PhaseAngDeg);
        if (!Is_Numeric($a) or FloatVal(trim($a)) > 360)  {return FALSE;}
   $a = abs(round(FloatVal($a),0));

   if ($a >= 0  and $a <= 10)
      {return 'New&nbsp;Moon.';}
   if ($a > 10  and $a <= 36)
      {return 'Waxing&nbsp;evening&nbsp;crescent,&nbsp;just&nbsp;past&nbsp;new.';}
   if ($a > 36  and $a <= 75)
      {return 'Waxing&nbsp;evening&nbsp;crescent&nbsp;moon.';}
   if ($a > 75  and $a <= 88)
      {return 'Waxing&nbsp;crescent,&nbsp;approaching&nbsp;first quarter.';}
   if ($a > 88  and $a <= 101)
      {return 'First&nbsp;quarter&nbsp;moon.';}
   if ($a > 101 and $a <= 115)
      {return 'Waxing&nbsp;gibbous,&nbsp;just past&nbsp;first&nbsp;quarter.';}
   if ($a > 115 and $a <= 159)
      {return 'Waxing&nbsp;gibbous&nbsp;moon.';}
   if ($a > 159 and $a <= 170)
      {return 'Waxing&nbsp;gibbous,&nbsp;approaching&nbsp;full&nbsp;moon.';}
   if ($a > 170 and $a <= 192)
      {return 'Full&nbsp;Moon.';}
   if ($a > 192 and $a <= 205)
      {return 'Waning&nbsp;gibbous,&nbsp;just&nbsp;past&nbsp;full&nbsp;moon.';}
   if ($a > 205 and $a <= 245)
      {return 'Waning&nbsp;gibbous&nbsp;moon.';}
   if ($a > 245 and $a <= 260)
      {return 'Waning&nbsp;gibbous,&nbsp;approaching&nbsp;last&nbsp;quarter.';}
   if ($a > 260 and $a <= 271)
      {return 'Last&nbsp;quarter&nbsp;moon.';}
   if ($a > 271 and $a <= 282)
      {return 'Waning&nbsp;crescent,&nbsp;just&nbsp;past&nbsp;last&nbsp;quarter.';}
   if ($a > 282 and $a <= 319)
      {return 'Waning&nbsp;morning&nbsp;crescent moon.';}
   if ($a > 319 and $a <= 349)
      {return 'Waning&nbsp;morning&nbsp;crescent,&nbsp;approaching&nbsp;new&nbsp;moon.';}
   if ($a > 349 and $a <= 360)
      {return 'New&nbsp;Moon.';}

} // End of  Lunar_Phase_to_Text (...)




/*
   ###########################################################################
   This function simply returns the technical details on using the API and is
   invoked when the ?! query is entered.

  Generic.  No special error checking is done.

  NO DEPENDENCIES
  ###########################################################################
*/

   function Get_Horizons_Tech_Data ()
{
// ===========================================================================
// Construct query URL for the NASA/JPL Horizons API.

   $Command = URLEncode("?!");

   $From_Horizons_API =
   "https://ssd.jpl.nasa.gov/api/horizons.api?format=text" .
   "&COMMAND='$Command'" .
   "&MAKE_EPHEM='NO'"    .
   "&OBJ_DATA='YES'"     ;

// ===========================================================================
// Send query to Horizons to obtain the technical details on using the API.

   $w = File_Get_Contents($From_Horizons_API);

   return HTMLEntities($w);

} // End of  Get_Horizons_Tech_Data()





   function Special_Horizons_Query ($Query)
{
   $Command = URLEncode(trim($Query));

   $From_Horizons_API =
   "https://ssd.jpl.nasa.gov/api/horizons.api?format=text" .
   "&COMMAND='$Command'" .
   "&MAKE_EPHEM='YES'"   .
   "&OBJ_DATA='NO'"      ;

// ===========================================================================
// Send query to Horizons API to obtain the basic physical data for the given
// body ID as a single long text string with formatting control codes.

   $W = File_Get_Contents($From_Horizons_API);

   return HTMLEntities($W);
}



/*
   ###########################################################################
   This function returns the Julian Day Number for any given calendar date in
   the range from BC 19999 to AD 19999 on the old Julian calendar or on the
   modern Gregorian calendar system.

   It returns negative JD numbers for dates prior to mathematical origins of
   the respective calendar systems.

   --------------------
   CALENDAR YEAR RANGE:

   BC 19999  to  AD 19999
   There is no calendar year 0 (zero).


   --------------------------------------------------------
   For the mathematical year number (y) used in calendrical
   computations for a BC calendar year Y = ('BC Year'):
   Y = 'BC 1949'  --->  y = '-1948'

   We take the numerical value  following the 'BC' prefix
   and subtract 1  and then change the result to negative
   to obtain the mathematical year value required for use
   in the (JDNum) computation.  This adjustment will only
   apply to 'BC' date strings.

   Example:  'BC 1949-May-20'
              y = -(Y - 1)
                = -(1949 - 1)
                = -1948
   So, (y,m,d)  = (-1948,5,20)


   -------------------------------------------------------------
   Positive calendar year numbers do NOT require any adjustment.
   Example:  'AD 1949-May-20'
   So, (y,m,d) = (1949,5,20)


   --------------------------------------------
   MATHEMATICAL ORIGINS OF THE CALENDAR SYSTEMS

   BC 4713-Jan-01-Mon   JDNum = 0   On the old Julian calendar
   BC 4714-Nov-24-Mon   JDNum = 0   on the modern Gregorian calendar

   NOTE:
   If a year is  given as a negative number,  it refers to a 'BC' year
   and will be converted to 'BC|AD' format internally for computations
   because 'BC' years first require a special numerical adjustment not
   needed for 'AD' years.

   Month = 1 to 12 or as 3-letter abbreviation string ('Jan' to 'Dec').
   Date strings are NOT case-sensitive.

   The returned signed JD Number is left-space-padded to 8 characters
   to facilitate easy columnar alignment if used for tabulation.

   ARGUMENTS:
   $BCADDateStr = Date string in BC|AD format.
                  VALID EXAMPLES:
                  'BC 9949-May-20'
                  'AD 1949-5-20'
                  '16959-1-29'
                  '-1023-Nov-15'

   JGAMode = Calendar mode to apply, where:
             'A' = Auto-select mode = Default
             'G' = Gregorian
             'J' = Julian

   ERRORS:
   FALSE is returned if an invalid argument is detected.

   NO DEPENDENCIES
   ###########################################################################
*/

   function JD_Num ($BCADDateStr, $JGAMode='A')
{
// --------------------------------------------------
// Read and adjust input date string argument format.

   $BCADDateStr = PReg_Replace("/\s+/", " ", trim($BCADDateStr));


/* ---------------------------------------------------
   If first character is a minus sign (negative year),
   then convert it into a 'BC' calendar year string.

   '-1949' becomes ---> 'BC 1949'
*/
   if (substr($BCADDateStr,0,1) == '-')
      {$BCADDateStr = 'BC ' . substr($BCADDateStr, 1, StrLen($BCADDateStr));}


// ---------------------------------------------------------------
// If no 'BC|AD' prefix at all, then attach a default 'AD' prefix.

   $ww = StrToUpper(substr($BCADDateStr,0,2));
   if ($ww <> 'BC' and $ww <> 'AD') {$BCADDateStr = "AD $BCADDateStr";}


// ------------------------------
// Read and parse date arguments.

   list($BCADYear,$Month,$Day) = PReg_Split("[-]", $BCADDateStr);


// ------------------
// A few adjustments.
   $BCADYear = trim($BCADYear);
   $Month    = trim($Month);
   $Day      = trim($Day);


// -----------------------------------
// Get BC|AD prefix and calendar year.

   $BCAD = StrToUpper(substr($BCADYear, 0,2));
   $Y = trim(substr($BCADYear, 2, StrLen($BCADYear)));


// ---------------------------------
// Adjust for BC year, if necessary.

   if ($BCAD == 'BC') {$Y = -$Y;}


// ------------------------------------------------------------
// Read calendar year argument value and return FALSE on error.

   $w = abs($Y);  if ($w < -19999 or $w == 0 or $w > 19999)  {return FALSE;}


// ---------------------------------------------------
// Read month argument. Could be a string or a number.

   $m = UCFirst(substr(StrToLower(trim($Month)),0,3));


// ------------------------
// Read day argument value.

   $d = trim($Day);


/* ----------------------------
  Read calendar mode argument.
  'G' = Gregorian | 'J' = Julian
  'A' = Auto-select = Default
*/
   $JGAMode = substr(StrToUpper(trim($JGAMode)),0,1);
   if ($JGAMode == '') {$JGAMode = 'A';}


// -------------------------------------------------
// Define abbreviations for month and weekday names.

   $MONTHS   = 'JanFebMarAprMayJunJulAugSepOctNovDec';
   $WEEKDAYS = 'SunMonTueWedThuFriSat';
   $JGDiff   = 0;


/* -----------------------------------------------------
   If month is a 3-letter abbreviation ('Jan' to 'Dec'),
   then replace it with the month number 1 to 12, if
   possible.  Otherwise, return FALSE if it cannot
   resolve the abbreviation text.
*/
   if (!Is_Numeric($m))
      {
       $i = StrPos($MONTHS, $m);
       if ($i === FALSE) {return $i;}
       $m = 1 + $i/3;
      }

// ---------------------------------------
// Error if invalid month number.

   if ($m < 1 or $m > 12) {return FALSE;}


/* -------------------------------------------------
   Proceed to compute the Julian calendar JD Number.
   This is the base JD Number value. If the Gregorian
   calendar is selected, then the difference between
   the calendars is applied to obtain the Gregorian
   calendar JD Number.
*/
   $A = floor((14-$m) / 12);
   $B = (($Y < 0)? $Y+1 : $Y) - $A;
   $C = floor($B/100);

   $JDNum = floor(30.6001*(12*$A + $m + 1))
          + floor(365.25*($B + 4716)) - 1524 + $d;


/* ----------------------------------------------
   Handle automatic calendar mode selection.
   If calendar mode = 'A' = Auto-select, then the
   calendar mode is automatically determined by
   the computed JD Number value.
*/

   if ($JGAMode == 'A')
      {
       $JGAMode = ($JDNum < 2299161)? 'J':'G';
      }


/* ---------------------------------------------
   Handle Gregorian (= default) calendar mode by
   by ADDING the difference  in days between the
   Julian and Gregorian JD Numbers, if indicated
   by the JGAMode setting.  This value could be
   negative or positive, depending on the given
   date.  Default Logic: If not 'J', then 'G'.

   GregorianJDNum = JulianJDNum + (+-JGDiff)
*/

   if ($JGAMode <> 'J')
      {
       $A = ($Y < 0)? $Y+1 : $Y;
       $B = trim($m);
       $C = $A - floor((14-$B) / 12);
       $D = floor($C/100);
       $JGDiff = (floor($D/4) - $D + 2);
      }
       $JDNum += $JGDiff;


/* -------------------------------------------------
   Error if (JDNum) is outside of the valid calendar
   range from 'BC 19999-Jan-01' to 'AD 19999-Dec-31'.
*/
   if ($JDNum < -5583211 or $JDNum > 9025909) {return FALSE;}


/* -----------------------------------------------
   Left-pad the signed JD number digits field with
   spaces to span exactly 8 characters width, just
   in case the values are used in a table.  This
   helps to arrange any (JDNum) column uniformly.

   The white space can be removed by performing
   a simple trim(JDNum) command, if not wanted.
*/
   $JDNum = SPrintF("% 8d", $JDNum);


// DONE:
   return $JDNum;

} // End of  JD_Num(...)










/*
   ###########################################################################
   This function is the inverse of the JD number function. Given any signed
   JD Number, it will return the corresponding calendar date string
   in 'BC|AD Yyyy-Mmm-dd-DoW' format.

   CALENDAR YEAR RANGE:
   BC 9999  to  AD 9999
   There is no calendar year 0 (zero).

   Mathematical origins of the calendar systems:
   BC 4713-Jan-01-Mon   JDNum = 0   On old Julian calendar
   BC 4714-Nov-24-Mon   JDNum = 0   On modern Gregorian calendar

   ARGUMENT:
   JDNumber = Julian Day number for the calendar date to be computed.

   JGAMode = Calendar mode
             'G' = Gregorian
             'J' = Julian
             'A' = Auto-select mode = Default

   RETURNS:
   Calendar date string in  'BC|AD Yyyy-Mmm-dd-DoW'  format
   according to the selected calendar mode.

   ERRORS:
   FALSE is returned if JD number argument is non-numeric.

   NO DEPENDENCIES
   ###########################################################################
*/

   function Inv_JD_Num ($JDNumber, $JGAMode='A')
 {
   $JDNum = trim($JDNumber);

// -------------------------------------------
// Define Month and Day-Of-Week abbreviations.

   $MONTHS   = 'JanFebMarAprMayJunJulAugSepOctNovDec';
   $WEEKDAYS = 'SunMonTueWedThuFriSat';

// ----------------------------
// Read calendar mode argument.
// 'G' = Gregorian = Default
// 'J' = Julian
// 'A' = Auto-select

   $JGAMode = substr(StrToUpper(trim($JGAMode)),0,1);
   if ($JGAMode == '') {$JGAMode = 'A';}

// ----------------------------------------------
// If calendar mode = 'A' = Auto-select, then the
// calendar mode is automatically determined by
// the given JD Number argument.

   if ($JGAMode == 'A')
      {
       $CMode  = ($JDNum < 2299161)? 0:1;
       $JGAMode = ($CMode == 0)? 'J':'G';
      }
   else
      {
       $CMode = ($JGAMode == 'J')? 0:1;
      }

// -----------------------------------------
// Compute numerical date elements (y, m, d)
// according to the calendar mode selection.

  $A = floor($JDNum + 0.5);
  $B = $CMode*floor(($A - 1867216.25) / 36524.25);
  $C = $A + $CMode*($B - floor($B/4) + 1);
  $D = $C + 1524;
  $E = floor(($D - 122.1) / 365.25);
  $F = floor(365.25 * $E);
  $G = floor(($D - $F) / 30.6001);
  $d = $D - $F - floor(30.6001 * $G);     // Day num   (1 to 31)
  $m = $G - 12*floor($G/14) - 1;          // Month num (1 to 12)
  $y = $E - 4716 + floor((14 - $m) / 12); // Mathematical year
  $Y = ($y > 0)? $y : $y-1;               // Calendar year (Negative = BC)

// ------------------------------------------------------------
// At this point we have the numerical date elements (Y, m, d).
// The next step is to construct the full calendar date text
// string for output. EXAMPLE OUTPUT: 'BC 9998-May-20-Tue'

   $i     = (7 + ($JDNum + 1) % 7) % 7;
   $DoW   = substr($WEEKDAYS, 3*$i, 3);
   $Y     = Str_Replace('-', 'BC ', SPrintF("%+05d", $Y));
   $Y     = Str_Replace('+', 'AD ', $Y);
   $Mmm   = substr($MONTHS, 3*($m-1), 3);
   $dd    = SPrintf("%02d", $d);
   $JDNum = SPrintF("% +8d", $JDNum);

// DONE.
   return "$Y-$Mmm-$dd-$DoW";

 } // End of  Inv_JD_Num(...)




/*
   ###########################################################################
   This function checks for an  invalid date string and returns boolean FALSE
   for an invalid date. If the date is OK, then it is returned in the same
   formalized format as 'BC 9949-Nov-01', 'AD 0149-May-20' or 'AD 0009-Jun-09'

   In formalized format, the year will always be expressed as 4 digits, padded
   by zeros as needed, like those in the above examples.

   NO DEPENDENCIES
   ###########################################################################
*/
   function Is_Valid_Date_Str ($BCADDateStr)
{
   $BCADDate = StrToUpper(trim($BCADDateStr));

// --------------------------------
// Define month name abbreviations.

   $MONTHS = 'JanFebMarAprMayJunJulAugSepOctNovDec';


// --------------------------------------------------------
// Account for a negative year input and change it to 'BC'.

   $BCAD = '';

   if (substr($BCADDate,0,1) == '-')
      {
       $BCADDate = 'BC '.substr($BCADDate,1,StrLen($BCADDate));
       list($BCAD, $YearMd) = PReg_Split("[ ]", $BCADDate);
       list($Year,$Month,$Day) = Preg_Split("[-]", $YearMd);
       $BCADDate = 'BC ' . SPrintF("%04d", $Year)."-$Month-$Day";
       $BCAD = 'BC';
      }

// ----------------------------------------------
// Error if missing any (-) separators.

   if (Substr_Count($BCADDate, '-') <> 2) {return FALSE;}



// -------------------------------------------------
// Break up date elements. Error if missing element.

   $wArray = PReg_Split("[-]", $BCADDate);
   $wCount = count($wArray);  if ($wCount <> 3){return FALSE;}



// -------------------------------------------
// Make year into a formal BCAD year string in
// the same format as 'AD 1776-Jul-04'

   $Year = StrToUpper(trim($wArray[0]));
   $Y = '';

   for($i=0;   $i < StrLen($Year);  $i++)
  {
   $CurrChar = substr($Year,$i,1);
   if (Is_Numeric($CurrChar)) {$Y .= $CurrChar;}
  }
   $FirstChar = substr($Year,0,1);

   if ($FirstChar == 'B') {$BCAD = 'BC';}
   if ($FirstChar == 'A') {$BCAD = 'AD';}
   if (Is_Numeric($Year))
      {
       $BCAD = ($Year < 0)? 'BC' : 'AD';
      }
   if ($BCAD == '') {return FALSE;} // Error if not BC or AD.

   $Year = "$BCAD ".SPrintF("%04d", $Y);


// -------------------------------------------------------
// Make numeric month into a 3-letter abbreviation string.

   $Month = UCFirst(StrToLower(substr(trim($wArray[1]),0,3)));

   if (!Is_Numeric($Month))
      {
       $i = StrPos($MONTHS, $Month);  if ($i === FALSE) {return $i;}
       $Month = 1 + $i/3;
       $m = $Month;
      }

   if (Is_Numeric($Month))
      {
       if ($Month < 1 or $Month > 12) {return FALSE;}
       $m = $Month;
       $Month = substr($MONTHS, 3*($Month-1), 3);
      }

   $Day = trim($wArray[2]);  if (!Is_Numeric($Day)) {return FALSE;}
     $d = $Day;  if ($d < 1 or $d > 31) {return FALSE;}
   $Day = SPrintF("%02d", $Day);



// --------------------------------------------------------------
// Read year argument. (BC converts to negative year internally).

   if (StrPos($Year, 'BC') !== FALSE)
      {$Y = -$Y;}



/* -------------------------------------------------------
   If negative year, then convert it to mathematical year.
   If positive, then no adjustment is needed.
*/
   $Y += ($Y < 1)? 1:0;



/* ---------------------------------------------------------
   Automatically determine leap year adjustment according to
   the 'J|G' = 'Julian|Gregorian' calendar mode switch.
   'J' = Julian calendar for all years up to AD 1582.
   'G' = Gregorian calendar for all years from AD 1583 onward.
*/

   $JG = ($Y < 1583)? 'J':'G';

   $LYFlag = ($JG == 'J')? $Y % 4 == 0 :
             ($Y % 4 == 0 and $Y % 100 <> 0) or $Y % 400 == 0;
   $LYAdj  = ($LYFlag)? 1:0;


// --------------------------
// Check for leap year error.

   if ($Month == 'Feb' and $Day == 29 and $LYFlag === FALSE) {return FALSE;}

   $FebDays = ($Month == 'Feb' and $Day == 29 and $LYFlag)? '29':'28';


// -----------------------------
// Check for day of month error.

   $mDays = substr("31$FebDays"."31303130313130313031", 2*($m-1), 2);
   if ($d > $mDays) {return FALSE;}


// -------------------------------------------------------
// Return reconstructed formal date string if all went OK.
// Otherwise, FALSE will be returned to indicate an error.

   return "$Year-$Month-$Day";

} // End of  Is_Valid_Date_Str (...)






/*
   ###########################################################################
   Given a BC|AD calendar date string, this function returns the corresponding
   mathematical date elements ('y,m,d') in a CSV format string.

   y = Mathematical year (0 = BC 0001)

   ARGUMENT:
   BCADDateStr = BC|AD Date String


   RETURNS:
   The numeric date elements ('Y,m,d') are returned in a CSV string.

   ERRORS:
   FALSE is returned if the given year is non-numeric.

   However, this function does NOT check the validity
   of the numbers it returns. It merely breaks the date
   string down into numbers, if in valid form and range.
*/

   function BCAD_Date_Str_to_Ymd ($BCADDateStr)
{
// Read and adjust input date string argument format.
   $BCADDateStr = PReg_Replace("/\s+/", " ", trim($BCADDateStr));

// Cut off any 3-letter week day abbreviation from end.
   $uuu = substr($BCADDateStr, -4);
   if (substr($uuu,0,1) == '-')
      {$BCADDateStr = substr($BCADDateStr,0, StrLen($BCADDateStr)-4);}


// If no BC|AD prefix, then attach default AD prefix.
   $ww = StrToUpper(substr($BCADDateStr,0,2));
   if ($ww <> 'BC' and $ww <> 'AD') {$BCADDateStr = "AD $BCADDateStr";}

// ------------------------------------
// Error if instant bad argument format.
   if (Substr_Count($BCADDateStr, '-') <> 2) {return FALSE;}

// ------------------------------
// Read and parse date arguments.
   list($BCADYear,$Month,$Day) = PReg_Split("[-]", $BCADDateStr);

// ---------------------------------------------------
// A few adjustments. Month can be a number or string.
   $BCADYear = trim($BCADYear);
   $Month    = $m = UCFirst(StrToLower(substr(trim($Month),0,3)));
   $Day      = $d = trim($Day);

// -----------------------------------
// Get BC|AD prefix and calendar year.
   $BCAD = StrToUpper(substr($BCADYear, 0,2));
   $Y = trim(substr($BCADYear, 2, StrLen($BCADYear)));

// --------------------------
// Error if Y is non-numeric.
   if (!Is_Numeric($Y))  {return FALSE;}

// ---------------------------------
// Adjust for BC year, if necessary.
// Y = Calendar year number (Neg = BC).
   if ($BCAD == 'BC') {$Y = -$Y;}

// ---------------------------------------------
// Error if year is 0 or outside calendar range.
// There is no year 0 (zero) on either calendar.
   if ($Y < -19999  or  $Y > 19999  or  $Y == 0)  {return FALSE;}

// -----------------------------------------------------
// If month is a 3-letter abbreviation ('Jan' to 'Dec'),
// then replace it with the month number 1 to 12, if
// possible.  Otherwise, return FALSE.

   if (!Is_Numeric($m))
      {
       $i = StrPos('JanFebMarAprMayJunJulAugSepOctNovDec', $m);
       if ($i === FALSE) {return $i;}
       $m = 1 + $i/3;
      }

   $m += 0;
   $d += 0;

// ---------------------------------------
// Error if invalid month number.
   if ($m < 1 or $m > 12) {return FALSE;}

// DONE.
   return "$Y, $m, $d";

} // End of  BCAD_Date_Str_to_Ymd (...)






/* ###########################################################################
   This function returns the number of days in any given month of any given
   year on the old Julian calendar or the modern Gregorian calendar.

   The month can be a number (1 to 12) or a 3-letter
   abbreviation ('Jan' to 'Dec').

   The 'AD' prefix is optional.

   -------------
   DEPENDENCIES:

   Is_Valid_Date()

   ###########################################################################
*/
   function Days_in_Month ($BCADDateStr)
{
   $BCADDate = trim($BCADDateStr);

   $W = Is_Valid_Date_Str($BCADDate);  if ($W === FALSE) {return FALSE;}

   $YmdStr = BCAD_Date_Str_to_Ymd ($W);

/* -----------------------------------------------------------
   Get 'Y/m/d' values corresponding to given BCAD date string.
   'BC' years will be expressed as negative values.  There is
   no year 0 (zero) on either calendar.    Note the numerical
   date string element sequence is 'Y/m/d'  and  NOT 'm/d/Y".
*/

   list($Y,$m,$d) = PReg_Split("[,]", $YmdStr);

   $JG = ($Y < 1583)? 'J':'G';

   $LYFlag = ($JG == 'J')? $Y % 4 == 0 :
             ($Y % 4 == 0 and $Y % 100 <> 0) or $Y % 400 == 0;
   $LYAdj  = ($LYFlag)? 1:0;

// --------------------------
// Check for leap year error.

   if ($m == 2 and $d == 29 and $LYFlag === FALSE) {return FALSE;}

   $FebDays = ($m == 2 and $d == 29 and $LYFlag)? '29':'28';

// ------------------------------------------------
// Check if day of month is outside of valid range.

   $mDays = substr("31$FebDays"."31303130313130313031", 2*($m-1), 2);
   if ($d > $mDays) {return FALSE;}

   return $mDays;
}





/*
   ###########################################################################
   ###########################################################################
   This function returns the simple lunar phase angle from 0 to 360 degrees
   based on the difference between the geocentric ecliptical longitudes of
   the moon and sun.  This gives a very good visual approximation to the
   geocentric lunar phase at the given local date, time and time zone.

   --------------------------------------------------------------
   Let:

   Lm = Geocentric ecliptical longitude of the moon (0 to 360 deg)
   Ls = Geocentric ecliptical longitude of the sun  (0 to 360 deg)

   PhaseAng = Simple lunar phase angle (0 to 360 deg)

   Then:

   w = 360 - Lm + Ls
   if (w > 360) then w = w - 360
   PhaseAng = 360 - w

   or equivalently:

   w = 360 - Lm + Ls
   PhaseAng = 360 - (w -= (w > 360)? 360:0)

   NO DEPENDENCIES
   ###########################################################################
*/

   function Lunar_Phase_Angle ($DateTimeStr, $TimeZone='+00:00', $DaySumYN='N')
{
   GLOBAL $Cnst;
   $DaySumYN  = substr(StrToUpper(trim($DaySumYN)),0,1);
   $DaySumAdj = ($DaySumYN == 'N')? 0:1;

/* ------------------------------------------------------
   Adjust for Daylight/Summer Time. This assumes that the
   Time Zone is given in the standard '+-HH:mm' format.
*/
   list($TZHH, $TZmm) = PReg_Split("[\:]", $TimeZone);
   $TZSign = substr($TZHH,0,1);
   $TZSignVal = ($TZSign == '-')? -1:1;
   $TZHours = $TZSignVal * (abs($TZHH) + $TZmm/60) + $DaySumAdj;
   $i = StrPos($TZHours, '.');
   if ($i == FALSE) {$TZHours .= '.00';}
   $i = StrPos($TZHours, '.');
   $TZHH = $TZSign.SPrintF("%02d", abs(substr($TZHours,0,$i)));
   $TimeZone = "$TZHH:$TZmm";

   $From_Horizons_API =
   "https://ssd.jpl.nasa.gov/api/horizons.api?format=text" .
   "&COMMAND='301'"                .
   "&OBJ_DATA='NO'"                .
   "&MAKE_EPHEM='YES'"             .
   "&EPHEM_TYPE='OBSERVER'"        .
   "&CAL_FORMAT='CAL'"             .
   "&CAL_TYPE='MIXED'"             .
   "&REF_SYSTEM='ICRF'"            .
   "&APPARENT='REFRACTED'"         .
   "&RANGE_UNITS='AU'"             .
   "&CENTER='500@399'"             .
   "&TIME_DIGITS='SECONDS'"        .
   "&TIME_ZONE='$TimeZone'"        .
   "&START_TIME='$DateTimeStr UT'" .
   "&STOP_TIME='$DateTimeStr.1'"   .
   "&STEP_SIZE='1d'"               .
   "&EXTRA_PREC='YES'"             .
   "&CSV_FORMAT='YES'"             .
   "&QUANTITIES='31,29"            ;

// ---------------------------------------------------------------------
// Extract the geocentric ecliptical longitude and latitude of the moon.

   $Moon = Str_Replace(",\n", " \n", trim(File_Get_Contents($From_Horizons_API)));

// ----------------------------------------------------
// Set pointers to start and end of ephemeris table and
// extract ONLY the required ephemeris CSV data line
// from between the Start/End pointers.

   $i    = StrPos($Moon, '$$SOE');
   $j    = StrPos($Moon, '$$EOE');
   $Moon = trim(substr($Moon, $i+5, $j-$i-5));


   $From_Horizons_API =
   "https://ssd.jpl.nasa.gov/api/horizons.api?format=text" .
   "&COMMAND='10'"                 .
   "&OBJ_DATA='NO'"                .
   "&MAKE_EPHEM='YES'"             .
   "&EPHEM_TYPE='OBSERVER'"        .
   "&CAL_FORMAT='CAL'"             .
   "&CAL_TYPE='MIXED'"             .
   "&REF_SYSTEM='ICRF'"            .
   "&APPARENT='REFRACTED'"         .
   "&RANGE_UNITS='AU'"             .
   "&CENTER='500@399'"             .
   "&TIME_DIGITS='SECONDS'"        .
   "&TIME_ZONE='$TimeZone'"        .
   "&START_TIME='$DateTimeStr UT'" .
   "&STOP_TIME='$DateTimeStr.1'"   .
   "&STEP_SIZE='1d'"               .
   "&EXTRA_PREC='YES'"             .
   "&CSV_FORMAT='YES'"             .
   "&QUANTITIES='31'"              ;

// --------------------------------------------------------------------
// Extract the geocentric ecliptical longitude and latitude of the sun.

   $Sun = Str_Replace(",\n", " \n", trim(File_Get_Contents($From_Horizons_API)));



/* ----------------------------------------------------
   Set pointers to start and end of ephemeris table and
   extract ONLY the required ephemeris CSV data line(s)
   from between the Start/End pointers.
*/
   $i   = StrPos($Sun, '$$SOE');
   $j   = StrPos($Sun, '$$EOE');
   $Sun = trim(substr($Sun, $i+5, $j-$i-5));


/* --------------------------------------------------------------
   Extract ONLY the ecliptical longitude values for moon and sun.
   The latitudes are not needed for these computations.
*/
   list($w,$w,$w, $Lm,$w,$Cnst) = PReg_Split("[,]", $Moon);
   list($w,$w,$w, $Ls) = PReg_Split("[,]", $Sun);

   $Cnst = trim($Cnst);

// --------------------------------------------------------
// Compute the simple lunar phase angle (0 to 360 degrees).

   $w = 360 - $Lm + $Ls;

   $PhaseAng = 360 - ($w -= ($w > 360)? 360:0);

   return $PhaseAng;

} // End of  Lunar_Phase_Angle (...)








?>





