<?php

/*
   ###########################################################################
   This program demonstrates a function to compute a table of lunar phase
   angles from (0 to 360 deg).

   It uses a cookie generated by the main input interface program settings for
   a general astrometric ephemeris*.

*  IMPORANT ASTROMETRIC EPHEMERIS QUANTITIES NOTE:
   For astrometric RA and Declination, use quantity #1 (NOT #2).
   Quantity #2 is for apparent geocentric or topocentric RA and Declination.

   Optional refraction does NOT apply to astrometric ephemerides.
   ###########################################################################
*/

   $PhasesLegend =
"           ==================================
            Phase_Deg    General_Description
           ===========  =====================
               0&deg;          New Moon
               45          Waxing Crescent
               90          First Quarter
              135          Waxing Gibbous
              180          Full Moon
              225          Waning Gibbous
              270          Last Quarter
              315          Waning Crescent
              360/0        New Moon
           ==================================";


// ----------------------------
// Get current existing cookie.

   $CookieName = 'NASA-JPL-Horizons-Ephemeris-Tool';

   $CookieDataString = Filter_Input(INPUT_COOKIE, $CookieName);
   list
  (
   $TargObjID,
   $TimeScale,
   $TimeZone,
   $StartBCAD,
   $StartYear,
   $StartMonth,
   $StartDay,
   $StartTime,
   $StopBCAD,
   $StopYear,
   $StopMonth,
   $StopDay,
   $StopTime,
   $StepSize,
   $LocName,
   $LonDeg,
   $LatDeg,
   $AltMet,
   $DaySumYN,
   $RefractYN,
   $DEGorHMS,
   $ObjDataYN,
   $SuppRangeRateYN,
   $JDateYN,
   $AUorKM,
   $EphemHeaderYN,
   $EphemFooterYN,
   $RefSystem,
   $GeocentYN,
   $Quantities
  ) = Preg_Split("[\|]", $CookieDataString);


// ----------------------------
// Construct Date/Time strings.

   $StartDateTime = "$StartYear-$StartMonth-$StartDay  $StartTime";
   $StopDateTime  = "$StopYear-$StopMonth-$StopDay  $StopTime";

   $HeaderYN = 'Yes';



/* --------------------------------------------
   Time zone offset in days. (TimeZone) MUST be
   be in normal signed '-+HH:mm' string format
   or it will not be read correctly.
*/
   $TZd = (substr($TimeZone,0,3) + substr($TimeZone, -2) / 60) / 24;



// Mysterious patch for abnormal DoW at 00:00
   $StepSize = PReg_Replace("/\s+/", " ", trim($StepSize));
   $StepUnits = substr(StrToLower(trim(StrRChr($StepSize, ' '))),0,2);




// --------------------
// Define Day/Sum note.

   $DaySumYNNote = ($DaySumYN == 'Yes')? "\nDaylight Saving / Summer":"\nStandard";


// -----------------------------
// Define TZ note for TT scale.

   $TTTZNoteTxt = "Local Time Zone   UT$TimeZone$DaySumYNNote Time";
   if ($TimeScale == 'TT')
      {
       $TTTZNoteTxt = "For Time Scale  = TT";
      }



// ----------------------------
// Define geocentric mode note.

   $GeocentTxt = (StrPos($GeocentYN, 'Yes') !== FALSE)? 'GEOCENTRIC':'';





// -------------------
// Call main function.

   $RawResult = Lunar_Phase_Table ($StartDateTime,$StopDateTime,$StepSize,
                                   $TimeZone,$DaySumYN,$HeaderYN);
// exit("[133]\n<pre>$RawResult</pre>");

   $TextArea1Text = $RawResult;







/* --------------------------------------------------------------------------
   Determine number of text columns and rows to use in the output text area.
   These values vary randomly according to the text block width and length.
   The idea is to eliminate the need for scroll-bars within the text area
   or worry as much about the variable dimensions of a text display area.
*/

// --------------------------------------------
// Text Area 1 - Default = At least 80 columns.

   $Text1Cols = 1 + Max(Array_Map('StrLen', PReg_Split("[\n]", trim($TextArea1Text))));
   $Text1Cols = 58; // Force to 58

//   if ($Text1Cols < 80) {$Text1Cols = 80;}  // Default
   $Text1Rows = 3 + Substr_Count($TextArea1Text, "\n");

// ---------------------------------
// Construct output table structure.

  $HTMLTableOut =
"<br>
<table bgcolor='#333333' cellspacing='1'>
<tr><td>
<pre style='background:white; color:black;'>
GEOCENTRIC LUNAR PHASE ANGLE TABLE

$TTTZNoteTxt

Optional Loc Name   = $LocName
Start Date/Time     = $StartDateTime
Stop  Date/Time     = $StopDateTime
Step Size           = $StepSize
</pre>
</td></tr>


<tr><td>

<span style='color:lime; font-size:10pt;'>Double-Click Within Text Area to Select ALL Text</span><br>
<textarea name='TextArea1' cols='$Text1Cols' rows='$Text1Rows' ReadOnly OnDblClick='this.select();' OnMouseUp='return true;'>
$RawResult
</textarea>
</td></tr></table>

<br><br><br><br><br><br><br><br><br><br><br><br>
<br><br><br><br><br><br><br><br><br><br><br><br>
";




print <<< _HTML

<!DOCTYPE HTML>

<HTML>

<head>
<title>Luna Phase Angle Table</title>

<style>
 BODY
{
 background:black; color:white; font-family:Verdana;
 font-size:10pt;
}

 TD
{
 background:black; color:silver; font-family:monospace;
 font-size:10.5pt; font-weight:bold; text-align:center;
 padding:8px;
}

 PRE
{
 background:black; color:silver; font-family:monospace;
 font-size:10.5pt; font-weight:bold; padding:4px;
 text-align:left; border-radius:8px;
}

 TEXTAREA
{
 background:white; color:black; font-family:monospace; font-size:10.5pt;
 font-weight:bold; padding:6pt; white-space:pre; border-radius:8px;
 line-height:125%;
}

::selection{background-color:yellow !important; color:black !important;}
::-moz-selection{background-color:yellow !important; color:black !important;}

</style>

</head>

<body>

$HTMLTableOut

</body>

</HTML>


_HTML;




/*
   ###########################################################################
   This function generates a table of geocentric lunar phases starting on any
   given date and spanning a given interval at regular time steps.

   It can also be used for a single computation for any given date and time.

   It takes into account the Time Zone and Daylight Saving/Summer Time.
   A table header is optional.

   ###########################################################################
*/

   function Lunar_Phase_Table ($StartDateTimeStr,$StopDateTimeStr,$StepSize,
                               $TimeZone='+00:00',$DaySumYN='No')
{
   $StartDateTimeStr = trim($StartDateTimeStr);
   $StopDateTimeStr  = trim($StopDateTimeStr);

/* -----------------------------------------------------------
   Adjust for Daylight/Summer Time, if indicated. This assumes
   that the Time Zone string is given in the standard +-HH:mm
   format or an error may occur.
*/
   $DaySumYN  = substr(StrToUpper(trim($DaySumYN)),0,1);
   $DSSTAdj = ($DaySumYN == 'N')? 0:1;


   $TZH = substr($TimeZone,0,3) . StrrChr((substr($TimeZone, -2)/60), '.');
   $TZH = $TZH + $DSSTAdj;



// **********************************************************
// MOON - GEOCENTRIC ECLIPTICAL LONGITUDE AND LATITUDE (GELL)

   $From_Horizons_API =
   "https://ssd.jpl.nasa.gov/api/horizons.api?format=text" .
   "&COMMAND='301'"                     .
   "&OBJ_DATA='NO'"                     .
   "&MAKE_EPHEM='YES'"                  .
   "&EPHEM_TYPE='OBSERVER'"             .
   "&CAL_FORMAT='BOTH'"                 .
   "&CAL_TYPE='MIXED'"                  .
   "&REF_SYSTEM='ICRF'"                 .
   "&APPARENT='AIRLESS'"                .
   "&RANGE_UNITS='AU'"                  .
   "&CENTER='500@399'"                  .
   "&TIME_DIGITS='SECONDS'"             .
   "&TIME_ZONE='$TZH'"                  .
   "&START_TIME='$StartDateTimeStr UT'" .
   "&STOP_TIME='$StopDateTimeStr'"      .
   "&STEP_SIZE='$StepSize'"             .
   "&EXTRA_PREC='YES'"                  .
   "&CSV_FORMAT='YES'"                  .
   "&QUANTITIES='31,29"                 ;

// -----------------------------------------------------------------
// Get the geocentric ecliptical longitude and latitude of the moon.

   $MoonGELL = Str_Replace(",\n", " \n", trim(File_Get_Contents($From_Horizons_API)));

/* ----------------------------------------------------
   Set pointers to start and end of ephemeris table and
   extract ONLY the required ephemeris CSV data lines
   from between the Start/End pointers.
*/
   $i = StrPos($MoonGELL, '$$SOE');
   $j = StrPos($MoonGELL, '$$EOE');

   $MoonTable = trim(substr($MoonGELL, $i+5, $j-$i-5));
   $MoonTable = (substr($MoonTable,0,1) <> 'b')?
                 " $MoonTable" : $MoonTable;





// *********************************************************
// SUN - GEOCENTRIC ECLIPTICAL LONGITUDE AND LATITUDE (GELL)

   $From_Horizons_API =
   "https://ssd.jpl.nasa.gov/api/horizons.api?format=text" .
   "&COMMAND='10'"                      .
   "&OBJ_DATA='NO'"                     .
   "&MAKE_EPHEM='YES'"                  .
   "&EPHEM_TYPE='OBSERVER'"             .
   "&CAL_FORMAT='BOTH'"                 .
   "&CAL_TYPE='MIXED'"                  .
   "&REF_SYSTEM='ICRF'"                 .
   "&APPARENT='AIRLESS'"                .
   "&RANGE_UNITS='AU'"                  .
   "&CENTER='500@399'"                  .
   "&TIME_DIGITS='SECONDS'"             .
   "&TIME_ZONE='$TimeZone'"             .
   "&START_TIME='$StartDateTimeStr UT'" .
   "&STOP_TIME='$StopDateTimeStr'"      .
   "&STEP_SIZE='$StepSize'"             .
   "&EXTRA_PREC='YES'"                  .
   "&CSV_FORMAT='YES'"                  .
   "&QUANTITIES='31,29"                 ;

// ----------------------------------------------------------------
// Get the geocentric ecliptical longitude and latitude of the sun.

   $SunGELL = Str_Replace(",\n", " \n", trim(File_Get_Contents($From_Horizons_API)));

/* ----------------------------------------------------
   Set pointers to start and end of ephemeris table and
   extract ONLY the required ephemeris CSV data lines
   from between the Start/End pointers.
*/
   $i = StrPos($SunGELL, '$$SOE');
   $j = StrPos($SunGELL, '$$EOE');

   $SunTable = trim(substr($SunGELL, $i+5, $j-$i-5));
   $SunTable = (substr($SunTable,0,1) <> 'b')? " $SunTable" : $SunTable;

/* ------------------------------------------
   PARSE THE TABLES AND CONSTRUCT A SINGLE
   CUSTOMIZED GEOCENTRIC LUNAR PHASE TABLE.

 2025-Feb-01  00:00:00  Aqr  2460707.708333333   35.36319
 ...

*/

   $LunarPhaseAngleTable = '';

/* --------------------------------------------
   Store lunar and solar ecliptical coordinates
   in their respective work arrays.
*/
   $MoonLonArray = PReg_Split("[\n]", $MoonTable);
   $MoonLonCount = count($MoonLonArray);

   $SunLonArray = PReg_Split("[\n]", $SunTable);
   $SunLonCount = count($SunLonArray);


/* -----------------------------------
   Fatal error if unequal array count.
*/
   if ($MoonLonCount <> $SunLonCount)
      {exit ("FATAL ERROR: Unequal array count.");}

/* -------------------------------------------
   Construct geocentric ecliptical coordinates
   and lunar phase table.
*/
   for ($i=0;   $i < $MoonLonCount;   $i++)
  {
   $CurrMoonLine = $MoonLonArray[$i];
   $CurrSunLine  = $SunLonArray[$i];

   list ($DateTimeStr,$JDate,$w,$w,$CurrMoonLon,$w, $MoonCnst) = PReg_Split("[,]", $CurrMoonLine);
   list ($w,$JDate,$w,$w,$CurrSunLon) = PReg_Split("[,]", $CurrSunLine);

   $DateTimeStr = Str_Replace(' ', '  ', trim($DateTimeStr));
   $DateTimeStr = (substr($DateTimeStr,0,1) <> 'b')? " $DateTimeStr" : $DateTimeStr;

// ----------------------------------------------------------
// Get IAU symbol for constellation in which moon is located.

   $MoonCnst = trim($MoonCnst);

// -------------------------------
// Get longitudes of moon and sun.

   $Lm = trim($CurrMoonLon);
   $Ls = trim($CurrSunLon);

// -------------------------------
// Compute the simple phase angle.

   $w = 360 - $Lm + $Ls;

   $PhaseAng = SPrintF("%1.4f", 360 - ($w -= ($w > 360)? 360:0));

   $PhaseAng = SPrintF("%8.4f", $PhaseAng);

// -----------------------------------------
// Construct current output table text line.

   $LunarPhaseAngleTable .= "$DateTimeStr  $MoonCnst $JDate  $PhaseAng\n";
  }
   $LunarPhaseAngleTable = RTrim($LunarPhaseAngleTable);

// Done.
   return $LunarPhaseAngleTable;

} // End of  Lunar_Phase_Table (...)





?>

