<?php

/*
   ###########################################################################
   This program demonstrates a function to generate a table of geocentric
   lunar phases at any given interval in any given time zone.

   It can generate a table for a single date and time or multiple dates at a
   given step interval.

   I uses a cookie generated by the main input interface program settings for
   an astrometric geocentric ephemeris.
   ###########################################################################
*/

// ----------------------------
// Get current existing cookie.

   $CookieName = 'NASA-JPL-Horizons-Ephemeris-Tool';

   $CookieDataString = Filter_Input(INPUT_COOKIE, $CookieName);
   list
  (
   $TargObjID,
   $TimeScale,
   $TimeZone,
   $StartBCAD,
   $StartYear,
   $StartMonth,
   $StartDay,
   $StartTime,
   $StopBCAD,
   $StopYear,
   $StopMonth,
   $StopDay,
   $StopTime,
   $StepSize,
   $LocName,
   $LonDeg,
   $LatDeg,
   $AltMet,
   $DaySumYN,
   $RefractYN,
   $DEGorHMS,
   $ObjDataYN,
   $SuppRangeRateYN,
   $JDateYN,
   $AUorKM,
   $EphemHeaderYN,
   $EphemFooterYN,
   $RefSystem,
   $GeocentYN,
   $Quantities
  ) = Preg_Split("[\|]", $CookieDataString);

   $StartDateTimeStr = "$StartBCAD $StartYear-$StartMonth-$StartDay  $StartTime";
   $StopDateTimeStr  = "$StopBCAD $StopYear-$StopMonth-$StopDay  $StopTime";

// --------------------------
// Compute lunar phase table.

   $w = Lunar_Phase_Table ($StartDateTimeStr,$StopDateTimeStr,$StepSize,
                           $TimeZone,$DaySumYN);

// exit("[65]\n<pre>$w</pre>");

// -----------------------------------------
// Store phase table lines in working array.

   $wArray = PReg_Split("[\n]", trim($w));
   $wCount = count($wArray);

   $TableRows = '';

   for($i=0;   $i < $wCount;   $i++)
  {
   $CurrLine = trim($wArray[$i]);
   $RightPart = substr($CurrLine, 0, StrLen($CurrLine)-10);
   $RightPart = PReg_Replace("/\s+/", " ", trim($RightPart));

   $uArray = PReg_Split("[\n]", trim($RightPart));
   $uCount = count($uArray);
   list($DateStr,$TimeStr,$CnstStr,$JDateUT) = PReg_Split("[ ]", $RightPart);
   $JDateUT = trim($JDateUT);

   $PhaseAng = trim(substr($CurrLine, -8));
   $PhaseNum = SPrintF("%03d", $PhaseAng + 0.5);
   $PhaseIMG = "$PhaseNum.png";
   $PhaseTxt = $PhaseNum + 0;

   $DaySumYNNote = ($DaySumYN == 'Yes')? "\nDaylight Saving / Summer":"\nStandard";

/* --------------------------------------------
   Time zone offset in days. (TimeZone) MUST be
   be in normal signed '-+HH:mm' string format
   or it will not read correctly.
*/
   $TZd = (substr($TimeZone,0,3) + substr($TimeZone, -2) / 60) / 24;


/* ---------------------------------------
   Compute local Julian Date and JD Number
   according to Time Zone.
*/

   $JDateLT = $JDateUT + $TZd;
   $JDNumLT = floor($JDateLT + 0.5);

/* -------------------------------------------------
   Compute local Day of Week (DoWStr) from (JDateLT)
   and Time Zone.
*/
   $DoW = ($JDNumLT + 1) % 7;
   $DoWStr = Substr('SunMonTueWedThuFriSat', 3*$DoW, 3);



   $TableRows .=
"<tr>
<td>$DateStr</td>
<td>$TimeStr</td>
<!-- <td>$JDateUT</td> --->
<td>$PhaseAng &deg;</td>
<td style='background:#000033;'><img src='moon-mini-yellow/$PhaseIMG' title=' $PhaseTxt &deg; '></td>
<td>$CnstStr</td>
</tr>\n";
  }

// -----------------------
// Special output filters.

   $TableRows = Str_Replace('b', 'BC ', $TableRows);
   $TableRows = Str_Replace('FeBC ', 'Feb', $TableRows);
   $TableRows = Str_Replace('LiBC', 'Lib', $TableRows);



  $HTMLTableOut =
"<br>
<span style='color:GreenYellow; font-size:13.5pt;'>Geocentric Daily Lunar Phase Table</span>
<br><br>
<pre>
Local Time Zone   UT$TimeZone$DaySumYNNote Time in Effect

Start Date/Time = $StartDateTimeStr
Stop  Date/Time = $StopDateTimeStr
Step Size       = $StepSize

</pre>
<br>
<table bgcolor='#333333' cellspacing='1'>
<tr>
<td style='background:#330000;' title=' Local date in the given time zone. '>Local_Date</td>
<td style='background:#003300;' title=' Local time in the given time zone. \n\n Could be Standard or Daylight / Summer Time. '>Local_Time</td>


<td style='background:#000044; text-align:center;'
title=' Simple Lunar Phase Angle.&nbsp;

Where:
  0&deg;   = New Moon
 45   = Waxing Crescent
 90   = First Quarter
135  = Waxing Gibbous
180  = Full Moon
225  = Waning Gibbous
270  = Last Quarter
315  = Waning Crescent
360  = New Moon \n\n'>
Phase_Angle</td>

<td style='background:#000000;' title=' Lunar Phase Image. '><span style='color:#C0C000;'><b>Image</b></span></td>
<td style='background:#222222;' title=' Constellation Symbol - IAU. '>Const</td>
</tr>
$TableRows
</table>
<br><br><br><br><br><br><br><br><br><br><br><br>
<br><br><br><br><br><br><br><br><br><br><br><br>
";




print <<< _HTML

<!DOCTYPE HTML>

<HTML>

<head>
<title>Geocentric Lunar Phase Table</title>

<style>
 BODY
{
 background:black; color:white; font-family:Verdana;
 font-size:10pt;
}

 TD
{
 background:black; color:white; font-family:monospace;
 font-size:11pt; font-weight:normal; text-align:center;
 padding:8px;
}

 PRE
{
 background:black; color:silver; font-family:monospace;
 font-size:10.5pt; font-weight:bold; padding:4px;
 text-align:left;
}

</style>

</head>

<body>

$HTMLTableOut

</body>

</HTML>


_HTML;




/*
   ###########################################################################
   This function generates a table of geocentric lunar phases starting on any
   given date and spanning a given interval at regular time steps.

   It can also be used for a single computation for any given date and time.

   It takes into account the Time Zone and Daylight Saving/Summer Time.
   A table header is optional.

   ###########################################################################
*/

   function Lunar_Phase_Table ($StartDateTimeStr,$StopDateTimeStr,$StepSize,
                               $TimeZone='+00:00',$DaySumYN='No')
{
   $StartDateTimeStr = trim($StartDateTimeStr);
   $StopDateTimeStr  = trim($StopDateTimeStr);

/* -----------------------------------------------------------
   Adjust for Daylight/Summer Time, if indicated. This assumes
   that the Time Zone string is given in the standard +-HH:mm
   format or an error may occur.
*/
   $DaySumYN  = substr(StrToUpper(trim($DaySumYN)),0,1);
   $DSSTAdj = ($DaySumYN == 'N')? 0:1;

   $TZH = substr($TimeZone,0,3) . StrrChr((substr($TimeZone, -2)/60), '.');
   $TZH = $TZH + $DSSTAdj;



// **********************************************************
// MOON - GEOCENTRIC ECLIPTICAL LONGITUDE AND LATITUDE (GELL)

   $From_Horizons_API =
   "https://ssd.jpl.nasa.gov/api/horizons.api?format=text" .
   "&COMMAND='301'"                     .
   "&OBJ_DATA='NO'"                     .
   "&MAKE_EPHEM='YES'"                  .
   "&EPHEM_TYPE='OBSERVER'"             .
   "&CAL_FORMAT='BOTH'"                 .
   "&CAL_TYPE='MIXED'"                  .
   "&REF_SYSTEM='ICRF'"                 .
   "&APPARENT='AIRLESS'"                .
   "&RANGE_UNITS='AU'"                  .
   "&CENTER='500@399'"                  .
   "&TIME_DIGITS='SECONDS'"             .
   "&TIME_ZONE='$TZH'"                  .
   "&START_TIME='$StartDateTimeStr UT'" .
   "&STOP_TIME='$StopDateTimeStr'"      .
   "&STEP_SIZE='$StepSize'"             .
   "&EXTRA_PREC='YES'"                  .
   "&CSV_FORMAT='YES'"                  .
   "&QUANTITIES='31,29"                 ;

// -----------------------------------------------------------------
// Get the geocentric ecliptical longitude and latitude of the moon.

   $MoonGELL = Str_Replace(",\n", " \n", trim(File_Get_Contents($From_Horizons_API)));

/* ----------------------------------------------------
   Set pointers to start and end of ephemeris table and
   extract ONLY the required ephemeris CSV data lines
   from between the Start/End pointers.
*/
   $i = StrPos($MoonGELL, '$$SOE');
   $j = StrPos($MoonGELL, '$$EOE');

   $MoonTable = trim(substr($MoonGELL, $i+5, $j-$i-5));
   $MoonTable = (substr($MoonTable,0,1) <> 'b')?
                 " $MoonTable" : $MoonTable;





// *********************************************************
// SUN - GEOCENTRIC ECLIPTICAL LONGITUDE AND LATITUDE (GELL)

   $From_Horizons_API =
   "https://ssd.jpl.nasa.gov/api/horizons.api?format=text" .
   "&COMMAND='10'"                      .
   "&OBJ_DATA='NO'"                     .
   "&MAKE_EPHEM='YES'"                  .
   "&EPHEM_TYPE='OBSERVER'"             .
   "&CAL_FORMAT='BOTH'"                 .
   "&CAL_TYPE='MIXED'"                  .
   "&REF_SYSTEM='ICRF'"                 .
   "&APPARENT='AIRLESS'"                .
   "&RANGE_UNITS='AU'"                  .
   "&CENTER='500@399'"                  .
   "&TIME_DIGITS='SECONDS'"             .
   "&TIME_ZONE='$TimeZone'"             .
   "&START_TIME='$StartDateTimeStr UT'" .
   "&STOP_TIME='$StopDateTimeStr'"      .
   "&STEP_SIZE='$StepSize'"             .
   "&EXTRA_PREC='YES'"                  .
   "&CSV_FORMAT='YES'"                  .
   "&QUANTITIES='31,29"                 ;

// ----------------------------------------------------------------
// Get the geocentric ecliptical longitude and latitude of the sun.

   $SunGELL = Str_Replace(",\n", " \n", trim(File_Get_Contents($From_Horizons_API)));

/* ----------------------------------------------------
   Set pointers to start and end of ephemeris table and
   extract ONLY the required ephemeris CSV data lines
   from between the Start/End pointers.
*/
   $i = StrPos($SunGELL, '$$SOE');
   $j = StrPos($SunGELL, '$$EOE');

   $SunTable = trim(substr($SunGELL, $i+5, $j-$i-5));
   $SunTable = (substr($SunTable,0,1) <> 'b')? " $SunTable" : $SunTable;

/* ------------------------------------------
   PARSE THE TABLES AND CONSTRUCT A SINGLE
   CUSTOMIZED GEOCENTRIC LUNAR PHASE TABLE.

 2025-Feb-01  00:00:00  Aqr  2460707.708333333   35.36319
 ...

*/

   $LunarPhaseAngleTable = '';

/* --------------------------------------------
   Store lunar and solar ecliptical coordinates
   in their respective work arrays.
*/
   $MoonLonArray = PReg_Split("[\n]", $MoonTable);
   $MoonLonCount = count($MoonLonArray);

   $SunLonArray = PReg_Split("[\n]", $SunTable);
   $SunLonCount = count($SunLonArray);


/* -----------------------------------
   Fatal error if unequal array count.
*/
   if ($MoonLonCount <> $SunLonCount)
      {exit ("FATAL ERROR: Unequal array count.");}

/* -------------------------------------------
   Construct geocentric ecliptical coordinates
   and lunar phase table.
*/
   for ($i=0;   $i < $MoonLonCount;   $i++)
  {
   $CurrMoonLine = $MoonLonArray[$i];
   $CurrSunLine  = $SunLonArray[$i];

   list ($DateTimeStr,$JDate,$w,$w,$CurrMoonLon,$w, $MoonCnst) = PReg_Split("[,]", $CurrMoonLine);
   list ($w,$JDate,$w,$w,$CurrSunLon) = PReg_Split("[,]", $CurrSunLine);

   $DateTimeStr = Str_Replace(' ', '  ', trim($DateTimeStr));
   $DateTimeStr = (substr($DateTimeStr,0,1) <> 'b')? " $DateTimeStr" : $DateTimeStr;

// ----------------------------------------------------------
// Get IAU symbol for constellation in which moon is located.

   $MoonCnst = trim($MoonCnst);

// -------------------------------
// Get longitudes of moon and sun.

   $Lm = trim($CurrMoonLon);
   $Ls = trim($CurrSunLon);

// -------------------------------
// Compute the simple phase angle.

   $w = 360 - $Lm + $Ls;

   $PhaseAng = SPrintF("%1.4f", 360 - ($w -= ($w > 360)? 360:0));

   $PhaseAng = SPrintF("%9.4f", $PhaseAng);

// -----------------------------------------
// Construct current output table text line.

   $LunarPhaseAngleTable .= "$DateTimeStr  $MoonCnst $JDate  $PhaseAng\n";
  }
   $LunarPhaseAngleTable = RTrim($LunarPhaseAngleTable);

// Done.
   return $LunarPhaseAngleTable;

} // End of  Lunar_Phase_Table (...)






?>

