<?php

/*
   ###########################################################################
   This program computes the Rise/Transit/Setting times of an astgronomical
   body for which a topocentric ephemeris is possible.

   It uses a cookie generated by the main input interface program settings for
   a general astrometric ephemeris*.

   ###########################################################################
*/

// -----------------------
// Include KERNAL module.

   include_once('KERNAL-2025.php');

// ----------------------------
// Get current existing cookie.

   $CookieName = 'NASA-JPL-Horizons-Ephemeris-Tool';

   $CookieDataString = Filter_Input(INPUT_COOKIE, $CookieName);
   list
  (
   $TargObjID,
   $TimeScale,
   $TimeZone,
   $StartBCAD,
   $StartYear,
   $StartMonth,
   $StartDay,
   $StartTime,
   $StopBCAD,
   $StopYear,
   $StopMonth,
   $StopDay,
   $StopTime,
   $StepSize,
   $LocName,
   $LonDeg,
   $LatDeg,
   $AltMet,
   $DaySumYN,
   $RefractYN,
   $DEGorHMS,
   $ObjDataYN,
   $SuppRangeRateYN,
   $JDateYN,
   $AUorKM,
   $EphemHeaderYN,
   $EphemFooterYN,
   $RefSystem,
   $GeocentYN,
   $Quantities
  ) = Preg_Split("[\|]", $CookieDataString);


   $GeoCentYN = 'No';

   $TargetBodyNameText = Get_Target_Name ($TargObjID);


// -----------------------
// Construct Date strings.

   $StartDate = "$StartBCAD $StartYear-$StartMonth-$StartDay";
   $StopDate  = "$StopBCAD $StopYear-$StopMonth-$StopDay";


// --------------------
// Define Day/Sum note.

   $DaySumYNNote = ($DaySumYN == 'Yes')? "Daylight Saving / Summer":"Standard";


// -------------------------------------------------------
// Define TZ note for TT scale and force to UT base scale.

   $TTTZNoteTxt = "Loc Time Zone   = UT$TimeZone  (+Pos = East)
$DaySumYNNote Time";
   if ($TimeScale == 'TT') {$TimeScale = 'UT';}

$OutputTextHeader =
"
$TargetBodyNameText

$TTTZNoteTxt

Start Date = $StartDate
Stop  Date = $StopDate

Optional Loc Name = $LocName
Longitude Deg     = $LonDeg (+Pos = East)
Latitude Deg      = $LatDeg
Alt Meters        = $AltMet m
";


// ------------------------------------
// Get the raw RTS ephemeris CSV table.

   $RawRTSTableCSV = R_T_S_Times ($TargObjID,$StartDate,$StopDate,$TimeZone,
                                  $DaySumYN,$LonDeg,$LatDeg,$AltMet);

// -------------------------------------------------
// Extract ONLY the CSV table body text lines block.

   $CSVTableExtract = Get_CSV_Body($RawRTSTableCSV);

   @$RTS = Make_RTS_Table ($CSVTableExtract,$OutputTextHeader);

// -------------------------------------
// Construct text block for (TextArea1).

   $TextArea1Text = $RTS;






/* --------------------------------------------
   Time zone offset in days. (TimeZone) MUST be
   be in normal signed '-+HH:mm' string format
   or it will not be read correctly.
*/
   $TZd = (substr($TimeZone,0,3) + substr($TimeZone, -2) / 60) / 24;



// Mysterious patch for abnormal DoW at 00:00
   $StepSize = PReg_Replace("/\s+/", " ", trim($StepSize));
   $StepUnits = substr(StrToLower(trim(StrRChr($StepSize, ' '))),0,2);




// --------------------
// Define Day/Sum note.

   $DaySumYNNote = ($DaySumYN == 'Yes')? 'Daylight Saving / Summer':'Standard';



/* --------------------------------------------------------------------------
   Determine number of text columns and rows to use in the output text area.
   These values vary randomly according to the text block width and length.
   The idea is to eliminate the need for scroll-bars within the text area
   or worry as much about the variable dimensions of a text display area.
*/



// -----------------------------
// Define TZ note for TT scale.

   $TTTZNoteTxt = "For Time Zone   UT$TimeZone&nbsp;&nbsp;&nbsp;($DaySumYNNote Time)";
   if ($TimeScale == 'TT')
      {
       $TTTZNoteTxt = "For Time Scale  = TT";
      }




// -------------------------------
// Set TextArea1 columns and rows.

   $Text1Cols = 63;
   $Text1Rows = 3 + Substr_Count($TextArea1Text, "\n");




// ---------------------------------
// Construct output table structure.

  $HTMLTableOut =
"<br>
<table bgcolor='#333333' cellspacing='1'>
<tr><td>

<span style='color:lime; font-size:10pt;'>Double-Click Within Text Area to Select ALL Text</span><br>
<textarea name='TextArea1' cols='$Text1Cols' rows='$Text1Rows' ReadOnly OnDblClick='this.select();' OnMouseUp='return true;'>
$TextArea1Text
</textarea>
</td></tr></table>

<br><br><br><br><br><br><br><br><br><br><br><br>
<br><br><br><br><br><br><br><br><br><br><br><br>
";




print <<< _HTML

<!DOCTYPE HTML>

<HTML>

<head>
<title>$TargetBodyNameText Rise/Transit/Set Times Ephemeris</title>

<style>
 BODY
{
 background:black; color:white; font-family:Verdana;
 font-size:10pt;
}

 TD
{
 background:black; color:silver; font-family:monospace;
 font-size:10.5pt; font-weight:bold; text-align:center;
 padding:8px;
}

 PRE
{
 background:black; color:silver; font-family:monospace;
 font-size:10.5pt; font-weight:bold; padding:4px;
 text-align:left;
}

 TEXTAREA
{
 background:white; color:black; font-family:monospace; font-size:10.5pt;
 font-weight:bold; padding:6pt; white-space:pre; border-radius:8px;
 line-height:125%;
}

::selection{background-color:yellow !important; color:black !important;}
::-moz-selection{background-color:yellow !important; color:black !important;}

</style>

</head>

<body>

$HTMLTableOut

</body>

</HTML>


_HTML;







/*
   ###########################################################################
   This function returns a topocentric ephemeris of rise/transit/set times
   from the NASA/JPL Horizons API. A topocentric ephemeris takes the location
   and altitude of the observer into account. Standard Earth refraction model
   is applied.

   DaySumYN = Means Daylght Saving/Summer Time 'Yes|No'
            'No'  = Use Standard Clock Time (Default)
            'Yes' = Use Daylight Saving/Summer Clock Time

   Angles are expressed in degrees.


   The test/demo ephemeris quantities returned are listed below, but more can
   easily be added.  The only extra feature added here applies to the Moon,
   appending two more columns (8,9) to the table.


COLUMN  CONTENT
======  =======================================================================
  1     Date and Time String (UT or ZONE)
  2     Sp (Solar Presence Symbol)
  3     Lp (Lunar Presence Symbol)
  4     Azimuth Angle (Compass Direction Reckoned Clockwise From North = 0)
  5     Elevation Angle (Relative to Horizon = 0 degrees)
  6     Apparent Visual Magnitude or Brightness Level
  7     S-brt (Brightness of 1 Square Arcsecond of Visible Disc)

  8     S-O-T (Sun-Observer-Target angle) used to find the lunar phase.
  9     /r (/L = Moon Leading the Sun  or  /T = Moon Trailing the Sun)


   NO DEPENDENCIES
   ###########################################################################
*/
   function R_T_S_Times ($BodyID,$StartDate,$StopDate,$TimeZone,$DaySumYN,
                         $LonDeg,$LatDeg,$AltMet)
{

// ===========================================================================
// Construct query URL for the NASA/JPL Horizons API.

   $BodyID     = trim($BodyID);
   $Command    = URLEncode($BodyID);
   $AltKm      = trim($AltMet) / 1000;

/* -----------------------------------------------------------
   Adjust for Daylight/Summer Time, if indicated. This assumes
   that the Time Zone string is given in the standard +-HH:mm
   format or an error may occur.
*/
   $DaySumYN  = substr(StrToUpper(trim($DaySumYN)),0,1);
   $DaySumAdj = ($DaySumYN == 'N')? 0:1;
   list($TZHH, $TZmm) = PReg_Split("[\:]", $TimeZone);
   $TZSign = substr($TZHH,0,1);
   $TZHours = (($TZSign == '-')? -1:1)*(abs($TZHH) + $TZmm/60) + $DaySumAdj;
   $i = StrPos($TZHours, '.');  if ($i == FALSE) {$TZHours .= '.00';}
   $i = StrPos($TZHours, '.');
   $TZHH = $TZSign.SPrintF("%02d", abs(substr($TZHours,0,$i)));
   $TimeZone = "$TZHH:$TZmm";

/* ----------------------------------------------------------
   Account for special case of the moon. Set distance (range)
   units = 'KM' and add SOT angle (Quantity #23) for finding
   the simple lunar phase angle.
*/
   $AUorKM = 'AU';  $SOT = ''; // Initialize.

   if ($BodyID == '301') {$AUorKM = 'KM';  $SOT = ',23';}

// -------------------------------------
// Construct URL for Horizons API query.

   $From_Horizons_API =
   "https://ssd.jpl.nasa.gov/api/horizons.api?format=text" .
   "&COMMAND='$Command'"                      .
   "&OBJ_DATA='YES'"                          .
   "&MAKE_EPHEM='YES'"                        .
   "&EPHEM_TYPE='OBSERVER'"                   .
   "&CAL_FORMAT='CAL'"                        .
   "&REF_SYSTEM='ICRF'"                       .
   "&APPARENT='REFRACTED'"                    .
   "&CENTER='COORD@399'"                      .
   "&COORD_TYPE='GEODETIC'"                   .
   "&SITE_COORD='$LonDeg,$LatDeg,$AltKm'"     .
   "&TIME_DIGITS='MINUTES'"                   .
   "&TIME_ZONE='$TimeZone'"                   .
   "&START_TIME='$StartDate%2000:00:00%20UT'" .
   "&STOP_TIME='$StopDate%2023:59:59'"        .
   "&STEP_SIZE='1 MINUTE'"                    .
   "&EXTRA_PREC='YES'"                        .
   "&R_T_S_ONLY='TVH'"                        .
   "&QUANTITIES='4,9$SOT'"                    .
   "&CSV_FORMAT='YES'"                        ;
// ============================================

/* -----------------------------------------------------------------------
   Send query to Horizons API to obtain the apparent topocentric ephemeris
   data we need for the R-T-S times of the given body ID.
*/
   $RTS = Str_Replace(",\n", " \n", File_Get_Contents($From_Horizons_API));

/* ----------------------------------------------------------------------
   If no ephemeris data is found, then return an empty string as an error
   state indicator.
*/
   if (StrPos($RTS, '$$SOE') === FALSE) {return $RTS;}

/* -------------------------------------------------------------------------
   DO NOT TRIM HERE BECAUSE INITIAL CHARACTER MAY BE A SPACE INDICATING AN
   'AD' YEAR.  'BC' YEARS BEGIN WITH A SINGLE 'b' AS THE INITIAL CHARACTER.

   --- DEAR NASA/JPL ---

   BAD PRACTICE. LEADING/TRAILING SPACES SHOULD NOT BE USED AS TABLE DATA.
   IT CAN SOMETIMES LEAD TO CONFUSION WHEN READING A LINE OF TABULAR TEXT
   IN DIFFERENT PROGRAMMING LANGUAGES. LEADING AND TRAINING SPACES MAY BE
   IGNORED.
   IF A DATA COLUMN IS EMPTY, A SPECIAL CHARACTER SHOULD BE ASSIGNED AS AN
   INDICATOR OF AN EMPTY COLUMN.  EXAMPLE (~), SIMILAR TO USING (n/a).
   SOMETHING OTHER THAN SIMPLY LEAVING IT EMPTY OR USING A SPACE CHARACTER
   ITSELF AS DATA.

   --- JUST A SUGGESTION ---
*/

   return $RTS;

} // End of  R_T_S_Times(...)




/*
   ###########################################################################
   This function constructs a complete Rise/Transit/Set Table from an RTStable
   ephemeris as returned by the Horizons API.

   EXTRACTED LINE EXAMPLE
Date__(UT)__HR:MN:SS, , ,Azimuth_(a-app), Elevation_(a-app),    APmag,  S-brt
   2025-Mar-09 10:08,*,r,   64.525919078,       0.059244577,   -0.067,  4.592

   GENERIC.  No special error checking is done.

   DEPENDENCY: Compass_Symbol()
   ###########################################################################
*/

   function Make_RTS_Table ($CSVTableExtract, $OutputTextHeader='')
{

   $OutputTextHeader = trim($OutputTextHeader);

   $RTS = ($CSVTableExtract);

   $wArray = PReg_Split("[\n]", $RTS);
   $wCount = count($wArray);

   $wTable = '';

   for ($i=0;   $i < $wCount;   $i++)
{
   $CurrLine = PReg_Replace("/\s+/", " ", $wArray[$i]);

// Parse line elements and store in variables.
   list (
         $DateTimeStr,
         $SP,
         $RorTorS,
         $Azim,
         $Elev,
         $ApMag,
         $SBrt
        ) = PReg_Split("[,]", $CurrLine);

// -----------------
// Trim white space.

   $DateTimeStr = Str_Replace(' ', '  ', trim($DateTimeStr));
   $RorTorS = trim($RorTorS);
   $Azim = trim($Azim);
   $Elev = trim($Elev);
   $ApMag = trim($ApMag);

// -----------------------
// Format output elements.

   if ($RorTorS == 'r') {$rts = 'Rise   ';}
   if ($RorTorS == 't') {$rts = 'Transit';}
   if ($RorTorS == 's') {$rts = 'Set    ';}

   $Azim  = SPrintF("% 7.3f",  $Azim);
   $Elev  = SPrintF("% +8.3f", $Elev);
   $ApMag = SPrintF("% +8.3f", $ApMag);


// ---------------------------
// Make special substututions.

   if ($RorTorS == 'r' or $RorTorS == 's') {$Elev = "    &#151;   ";}
   if ($RorTorS == 't') {$Azim = '   |   ';}

   $CompassSymbol = ($RorTorS == 't')?  ' | ' : Compass_Symbol($Azim);

// -------------------------------------------------------
// Determine if blank line is needed and apply it, if so.

   $wTable .= "$DateTimeStr  $rts  $Azim   $CompassSymbol  $Elev  $ApMag\n";
}

// ----------------------------------------------
// Split by days using blank lines between dates.

   $wArray = PReg_Split("[\n]", trim($wTable));
   $wCount = count($wArray);
   $wTable = trim($wArray[0]) . "\n";

   for ($i=1;   $i< $wCount;   $i++)
  {
   $PrevLine = trim($wArray[$i-1]);
   $ddPrev = substr($PrevLine,9,2);

   $CurrLine = trim($wArray[$i-0]);
   $ddCurr = substr($CurrLine,9,2);

   $BlankLine = ($ddCurr == $ddPrev)? "":"\n";

   $wTable .= "$BlankLine$CurrLine\n";
  }

   if (StrPos($wTable, 'Cannot  interpret  date.') !== FALSE)
      return "$StartDate\nCheck the dates for error(s).";



   $OutputHeader =
"
TOPOCENTRIC Rise/Transit/Set Times (UT or Local Zone Time Only)

Target Object   = $OutputTextHeader

==============================================================
Calendar Date/Time   Event     Azim    Dir     Elev    Vis_Mag
==================  =======  =======   ===   =======   =======\n";

   return "$OutputHeader$wTable";

} // End of  Make_RTS_Table (...)






?>

